/**
  ********************************************************************
  * @file project\main.c
  * @brief This file contains the firmware main function.
  * @STMicroelectronics - RFID Application Team
  * @version V1.1.0
  ********************************************************************
  */

/* Includes --------------------------------------------------------*/
#include "stm8l10x.h"
#include "stm8l10x_clk.h"
#include "stm8l10x_gpio.h"
#include "stm8l10x_awu.h"
#include "stm8l10x_tim2.h"
#include "stm8l10x_i2c.h"
#include "stdio.h"
#include "stdlib.h"

/*private i2c library for both Dual eeprom and temperature sensor */
#include "i2c_ee.h"     

/* Private defines -------------------------------------------------*/
#define BUFFER_SIZE ((uint8_t)2)
#define START   0x11 /* Predefined System Status */
#define PAUSED  0x22 /* Predefined System Status */
#define RUNNING 0x33 /* Predefined System Status */
#define STOPPED 0x44 /* Predefined System Status */
#define UPDATE  0x55 /* Predefined System Status */
#define OTHER   0x66 /* Predefined System Status */

/* Private function prototypes -------------------------------------*/
void Delay(uint16_t nCount); /* to store the acquisition freqency */
void AWU_AutoLSICalibration(void);

ErrorStatus start_acquisition   (void); /*return value of the function */
ErrorStatus stop_acquisition    (void); /*return value of the function */
ErrorStatus acquisition_running (void); /*return value of the function */
ErrorStatus acquisition_update  (void); /*return value of the function */

/* Global variable -------------------------------------------------*/
uint8_t  Status         = 0x00; 					/* SYSTEM BYTE */
uint8_t  Delay_Sleep    = 0x13; 					/* SYSTEM BYTE */
uint8_t  OverWrite      = 0x00;						/* SYSTEM BYTE */
uint8_t  Nb_Temperature[2] = {0x00,0x00};	/* SYSTEM BYTE */

uint16_t FREE_SPACE     = 0x0FFC; /* 4092 values */
uint16_t POINTER_ACTUAL = 0x0008; /* first data will be writen here */
uint16_t Status_Add     = 0x0000; /* address of the SYSTEM BYTE */
uint16_t OverWrite_Add  = 0x0001; /* address of the SYSTEM BYTE */
uint16_t Delay_Add      = 0x0002;	/* address of the SYSTEM BYTE */
uint16_t Nb_Temp_Add    = 0x0004; /* address of the SYSTEM BYTE */

ErrorStatus ErrorCode;
ErrorStatus ErrorFunction;

/********************************************************************/
/*---------------------------- MAIN --------------------------------*/
/********************************************************************/
void main(void)
{
	/*----------------------  CLOCKS CONFIG -----------------------*/
	CLK_DeInit();/*all*/			
	
	CLK_MasterPrescalerConfig(CLK_MasterPrescaler_HSIDiv8);/*2MHz*/
	CLK_PeripheralClockConfig(CLK_Peripheral_I2C, ENABLE);/*I2C*/
	CLK_PeripheralClockConfig(CLK_Peripheral_TIM2,ENABLE);/*AWU*/
	CLK_PeripheralClockConfig(CLK_Peripheral_AWU, ENABLE);/*AWU*/

	/*-----------------------------  GPIO --------------------------*/
	
	/* Initialization of I/Os in Input Mode to minimize comsumption*/
  GPIO_Init(GPIOA, (GPIO_Pin_1 |GPIO_Pin_2 | GPIO_Pin_3 | GPIO_Pin_4 | GPIO_Pin_5 |GPIO_Pin_6),GPIO_Mode_In_PU_No_IT );
	
	GPIO_Init(GPIOB, (GPIO_Pin_0 |GPIO_Pin_1 |GPIO_Pin_2|GPIO_Pin_3 | GPIO_Pin_4 | GPIO_Pin_5 | GPIO_Pin_6| GPIO_Pin_7),GPIO_Mode_In_PU_No_IT );
	
	GPIO_Init(GPIOC, (GPIO_Pin_2 | GPIO_Pin_3| GPIO_Pin_4| GPIO_Pin_5| GPIO_Pin_6),GPIO_Mode_In_PU_No_IT );
	
	GPIO_Init(GPIOD, (GPIO_Pin_0 | GPIO_Pin_1 | GPIO_Pin_2 | GPIO_Pin_3|GPIO_Pin_4 ),GPIO_Mode_In_PU_No_IT );
	
	/* Initialization of I/Os in Output Mode */
	/* To power dual eeprom & sensor temperature & Vcc Bus i2c */
	GPIO_Init(GPIOD, GPIO_Pin_5| GPIO_Pin_6 |GPIO_Pin_7, GPIO_Mode_Out_PP_Low_Fast);
	
	AWU_AutoLSICalibration();   /* measure the LSI clock frequency */
	AWU_Init(AWU_Timebase_1s); 	/* selection of time base */
	AWU_Cmd(ENABLE);					  /* set the AWUEN bit */
	enableInterrupts();	


	while (1) /* routine start */
	{
		GPIO_SetBits(GPIOD, GPIO_Pin_5);/*power Vcc bus i2c*/
	  GPIO_SetBits(GPIOD, GPIO_Pin_7);/*power dual eeprom*/

		if(Status == 0x00) /* Fisrt Boot */
		{
			ErrorFunction = stop_acquisition();
		}
		else
		{
			I2C_EEInit_EEPROM(); /* init eeprom dual*/
			ErrorFunction = I2C_EE_BufferRead(&Status, Status_Add, 0x01);
			if(ErrorFunction!=ERROR)
			{
				if(Status == START)
				{
					ErrorFunction = start_acquisition();
				}
				
				else if(Status == RUNNING)
				{
						ErrorFunction = acquisition_running();
				}
				
				else if(Status == UPDATE)
				{
					ErrorFunction = acquisition_update();
				}
				
			}
		}
		
		GPIO_ResetBits(GPIOD, GPIO_Pin_6);/*Unpower temperature sensor*/
		GPIO_ResetBits(GPIOD, GPIO_Pin_7);/*Unpower dual eeprom*/
		GPIO_ResetBits(GPIOD, GPIO_Pin_5);/*Unpower Vcc bus i2c*/

		halt();/*Switch in Active-Halt mode for a defined Delay*/
		
	}
}

/**************************************************************/
/************************ START *******************************/
/**************************************************************/
ErrorStatus start_acquisition(void)
{
	/* reset system variable */
	POINTER_ACTUAL    = 0x0008;
	Nb_Temperature[0] = 0x00;
	Nb_Temperature[1] = 0x00;
	FREE_SPACE        = 0x0FFC; /* 4092 values */
	Status            = RUNNING;
	Delay_Sleep    		= 0xd; /* 1 second */
	
	I2C_EEInit_EEPROM(); /* init dual eeprom */
	ErrorCode = I2C_EE_PageWrite(&Delay_Sleep, Delay_Add, 0x01);/* READ DELAY */
	Delay((uint16_t)2000);
	if(ErrorCode != ERROR)
	{
		I2C_EEInit_EEPROM(); /* dual init eeprom */
		ErrorCode = I2C_EE_BufferRead(&Delay_Sleep, Delay_Add, 0x01);/* READ DELAY */
		if(ErrorCode != ERROR)
		{
			AWU_Init((uint8_t)Delay_Sleep); /* selection of time base */
			AWU_Cmd(ENABLE);					  		/* set the AWUEN bit */
			
			I2C_EEInit_EEPROM();
			ErrorCode = I2C_EE_PageWrite(&Status, Status_Add, 0x01);
			if(ErrorCode != ERROR)
			{
				Delay((uint16_t)2000);
				
				I2C_EEInit_EEPROM();
				ErrorCode = I2C_EE_PageWrite(Nb_Temperature, Nb_Temp_Add, 0x02);
				if(ErrorCode != ERROR)
				{
					Delay((uint16_t)2000);
				}
			}
		}
	}
	return ErrorCode;
}

/**************************************************************/
/************************ RUNNING *****************************/
/**************************************************************/
ErrorStatus acquisition_running(void)
{
	uint8_t  data_sensor[2] = {0x00,0x00}; /* to store temperature */
	uint8_t  Pointer_temperature = 0x00;   /* temperature access */
	uint16_t ConfigBytes = 0x0183;         /* sensor OneShotMode 9bit */
	
	GPIO_SetBits(GPIOD, GPIO_Pin_6);/* Power sensor */
	I2C_EEInit_SENSOR(); /* int EEPROM & SENSOR */
	
	if(FREE_SPACE == 0x0000)/* FULL MEMORY */
	{
		ErrorCode = I2C_EE_BufferRead(&OverWrite, OverWrite_Add, 0x01);/* READ OVERWRITE */
		if(ErrorCode != ERROR)
		{
			if(OverWrite == 0x11)/* Authorized */
			{
				ErrorFunction = start_acquisition();
			}
			else /* Non Authorized */
			{
				ErrorFunction = stop_acquisition();
			}
		}
	}
		
	else
	{
		ErrorCode = I2C_SS_Config(ConfigBytes);/* Config the Temperature sensor in OneShotMode */
		if(ErrorCode != ERROR)
		{
			ErrorCode = I2C_SS_Config(ConfigBytes);/* Get a temperature */
			if(ErrorCode != ERROR)
			{
				Delay((uint16_t)9000);     /* TEMPO */
				
				/* read 2bytes temperature and store them into data_sensor[] */
				ErrorCode =I2C_SS_BufferRead(data_sensor,Pointer_temperature,0x02);
				if(ErrorCode != ERROR)
				{
					Delay((uint16_t)9000);     /* TEMPO */
					
					I2C_EEInit_EEPROM();
					ErrorCode = I2C_EE_PageWrite(data_sensor, POINTER_ACTUAL,0x02);/* write dual eeprom */
					if(ErrorCode != ERROR)
					{
						Delay((uint16_t)2000); 	/* to let the time writting to dual eeprom */
						POINTER_ACTUAL += 2;
						FREE_SPACE --;
						
						if(Nb_Temperature[0]==0xFF)
						{
							Nb_Temperature[1]++;
							Nb_Temperature[0]=0x00;
						}
						else
						{
							Nb_Temperature[0] ++;
						}
						
						I2C_EEInit_EEPROM();
						ErrorCode = I2C_EE_PageWrite(Nb_Temperature, Nb_Temp_Add ,0x02);/*ECRITURE EE*/
						if(ErrorCode != ERROR)
						{
							Delay((uint16_t)2000);
						}
					}
				}
			}
		}
	}
	return ErrorCode;
}

/**************************************************************/
/************************ STOP ********************************/
/**************************************************************/
ErrorStatus stop_acquisition(void)
{
	unsigned char Status_temp= STOPPED;
	Status = OTHER ;
	
	I2C_EEInit_EEPROM(); /* dual init eeprom */
	if(I2C_EE_PageWrite(&Status_temp, Status_Add, 0x01) != ERROR)
	{
		Delay((uint16_t)2000);
		ErrorCode = SUCCESS;
	}
	else
	{
		ErrorCode = ERROR;
	}
	
	AWU_Init(AWU_Timebase_2s); 	/* selection of time base */
	AWU_Cmd(ENABLE);	
	enableInterrupts();
	
	return ErrorCode;
}

/**************************************************************/
/************************ UPDATE ******************************/
/**************************************************************/
ErrorStatus acquisition_update  (void)
{
	unsigned char Status_temp= RUNNING;
	
	I2C_EEInit_EEPROM(); /* init dual eeprom */
	ErrorCode = I2C_EE_BufferRead(&Delay_Sleep, Delay_Add, 0x01);/* READ DELAY */
	if(ErrorCode != ERROR)
	{
		/*-----------------------------  AWU --------------------------*/
		AWU_Init((uint8_t)Delay_Sleep); /* selection of time base */
		AWU_Cmd(ENABLE);					  /* set the AWUEN bit */
	}
	
	I2C_EEInit_EEPROM(); /* init eeprom dual*/
	if(I2C_EE_PageWrite(&Status_temp, Status_Add, 0x01) != ERROR)
	{
		Delay((uint16_t)2000);
		ErrorCode = SUCCESS;
	}
	else
	{
		ErrorCode = ERROR;
	}
	return ErrorCode;
}

void Delay(uint16_t nCount)
{
	/* Decrement nCount value */
	while (nCount != 0)
	{
			nCount--;
	}
}

void AWU_AutoLSICalibration(void)
{

  uint32_t lsi_freq_hz = 0x0;
  uint32_t fmaster = 0x0;
  uint16_t ICValue1 = 0x0;
  uint16_t ICValue2 = 0x0;

  /* Get master frequency */
  fmaster = CLK_GetClockFreq();

	/* Measure the LSI frequency with TIMER Input Capture 1 */
	/* Enable the LSI measurement: LSI clock connected to timer Input Capture 1 */
  AWU->CSR |= AWU_CSR_MSR;

	/* Capture only every 8 events!!! */
	TIM2_ICInit (	TIM2_Channel_1 , TIM2_ICPolarity_Rising , TIM2_ICSelection_DirectTI , TIM2_ICPSC_Div8 , 0x0);

	/* Enable TIM2 */
	TIM2_Cmd(ENABLE);

	TIM2->SR1 = 0x00;
	TIM2->SR2 = 0x00;

	/* Clear CC1 Flag*/
	TIM2_ClearFlag(TIM2_FLAG_CC1);

	/* wait a capture on cc1 */
	while ((TIM2->SR1 & (uint8_t)TIM2_FLAG_CC1) != TIM2_FLAG_CC1)
	{
	}
	/* Get CCR1 value*/
	ICValue1 = TIM2_GetCapture1();
	TIM2_ClearFlag(TIM2_FLAG_CC1);

	/* wait a capture on cc1 */
	while ((TIM2->SR1 & (uint8_t)TIM2_FLAG_CC1) != TIM2_FLAG_CC1)
	{
	}
	/* Get CCR1 value*/
	ICValue2 = TIM2_GetCapture1();
	TIM2_ClearFlag(TIM2_FLAG_CC1);

	/* Disable IC1 input capture */
	TIM2->CCER1 &= (uint8_t)(~TIM_CCER1_CC1E);
	/* Reset CCMR1 register */
	TIM2->CCMR1 = 0x00;
	/* Disable TIM2 */
	TIM2_Cmd(DISABLE);
	
	/* Compute LSI clock frequency */
  lsi_freq_hz = (8 * fmaster) / (ICValue2 - ICValue1);

	/* Disable the LSI measurement: LSI clock disconnected from timer Input Capture 1 */
  AWU->CSR &= (uint8_t)(~AWU_CSR_MSR);

  /* Update the calibration registers */
  AWU_LSICalibrationConfig(lsi_freq_hz);
}


#ifdef  USE_FULL_ASSERT

/**
  * @brief  Reports the name of the source file and the source line number
  *   where the assert_param error has occurred.
  * @param file: pointer to the source file name
  * @param line: assert_param error line source number
  * @retval : None
  */
void assert_failed(uint8_t* file, uint32_t line)
{ 
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
		/* allumer la LED rouge */
  }
}
#endif

/********* (C) COPYRIGHT 2009 STMicroelectronics *****END OF FILE****/
